/*****************************************************************************
* @file     TestCode_ADC_DmaP2P.c
* @brief    Contains the external function prototypes for TestCode
* @author   AE Team, ABOV Semiconductor Co., Ltd.
* @version  V0.0.1
* @date     16. MAY. 2019
*
* Copyright(C) 2019, ABOV Semiconductor
* All rights reserved.
*
*
********************************************************************************
* DISCLAIMER
*
* THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
* TIME. AS A RESULT, ABOV SEMICONDUCTOR DISCLAIMS ALL LIABILITIES FROM ANY
* DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
* FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
* CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*
*******************************************************************************/

/*******************************************************************************
* Pre-processor Definition & Macro
*******************************************************************************/
#include "TestCode_ADC_DmaP2P.h"
#include "main_conf.h"

/*******************************************************************************
* Private Typedef
*******************************************************************************/

/*******************************************************************************
* Private Variable
*******************************************************************************/
static uint32_t MilliSec;
static ADC_CFG_Type ADC_Config;

static Bool DoneFlag;
static Bool DMA_Done;

#define DMA_DATA_SIZE		(1)
#define DMA_ADDR_PERI 	(0x40003450UL) // Peripheral Register for DMA Peri-to-Peri
#define DMA_ADDR_TARGET (ADC->TRG)	 // DMA Target Address for DMA Tx Test

#define DMA_TEST_CH			(DMAC0)
//#define DMA_TEST_CH			(DMAC1)
//#define DMA_TEST_CH			(DMAC2)
//#define DMA_TEST_CH			(DMAC3)
//#define DMA_TEST_CH			(DMAC4)
//#define DMA_TEST_CH			(DMAC5)
//#define DMA_TEST_CH			(DMAC6)
//#define DMA_TEST_CH			(DMAC7)

/*******************************************************************************
* Private Function Prototype
*******************************************************************************/

/*******************************************************************************
* Public Function
*******************************************************************************/
/*******************************************************************************
* @brief      SysTick interrupt handler @ 1000Hz(1ms)
* @param      None
* @return     None
*******************************************************************************/
void SysTick_Handler_ADC_DmaP2P(void) {
	if (MilliSec) {
		MilliSec--;
	}
}

/*******************************************************************************
* @brief      ADC interrupt handler
* @param      None
* @return     None
*******************************************************************************/
void ADC_IRQHandler_ADC_DmaP2P(void) {
	ADC_STATUS_Type status; // check interrupt flag

	status = HAL_ADC_GetStatus(ADC);

	DoneFlag = TRUE;

	// EOC Flag
	if (status & ADC_STATUS_SINGLE_INT) {
		HAL_ADC_ClearStatus(ADC, ADC_STATUS_SINGLE_INT);
	}
	// EOS Flag
	if (status & ADC_STATUS_SEQUENCE_INT) {
		HAL_ADC_ClearStatus(ADC, ADC_STATUS_SEQUENCE_INT);
	}
	// Trigger Flag
	if (status & ADC_STATUS_TRIGGER_INT) {
		HAL_ADC_ClearStatus(ADC, ADC_STATUS_TRIGGER_INT);
	}

	// DMA Done Flag
	if (status & ADC_STATUS_DMA_DONE) {
		DMA_Done = TRUE;
		HAL_ADC_ClearStatus(ADC, ADC_STATUS_DMA_DONE);
		HAL_DMAC_SetTransferCount(DMA_TEST_CH, DMA_DATA_SIZE);
	}
	// DMA Overrun Flag
	if (status & ADC_STATUS_DMA_OVER) {
		HAL_ADC_ClearStatus(ADC, ADC_STATUS_DMA_OVER);
	}
}

/*******************************************************************************
* @brief      Main_AdcPeri2Peri
* @param      None
* @return     None
*******************************************************************************/
void Main_AdcPeri2Peri(void) {
	/* Outputs DAC data from measured ADC data via Peripheral-to-Peripheral of DMA */

	uint32_t ADC_Data;
    uint32_t DAC_Data;
	DMAC_CFG_Type DMA_Config;
	DAC_CFG_Type DAC_Config; 
    
	// DAO(PA6) setting
	HAL_GPIO_ConfigOutput(PA, 6, PCU_MODE_ALT_FUNC);
	HAL_GPIO_ConfigFunction(PA, 6, PCU_ALT_FUNCTION_3);
	HAL_GPIO_ConfigPullup(PA, 6, PCU_PUPD_DISABLE);

	// DAC setting
	DAC_Config.Stanby = DISABLE;									// DAC normal mode
	DAC_Config.DAC_Out2 = DISABLE;									// Disconnect with comparator input
	DAC_Config.DAC_Out1 = DISABLE;									// Disconnect with ADC ch.20
	DAC_Config.DAC_Out0 = ENABLE;									// Connect with DAO(PA6)
	DAC_Config.OutBuffer = ENABLE;									// Enable Buffer to output DAC pin
	DAC_Config.ReloadSignal = DAC_RELOAD_SIGNAL_ALWAYS;				// DAC reload signal
	HAL_DAC_Init(DAC, &DAC_Config);									// Initialized DAC Configuration

	// Initialize DAC data (Gain = 0DB, No Offset)
	ADC_Data = 0x00;
    DAC_Data = 0x00;
	HAL_DAC_SetData(DAC, (uint16_t)(ADC_Data << 4));									// Set DAC Data Value (0x0)
	HAL_DAC_Start(DAC);																	// Start DAC Operation

	// Set ADC input pin
	HAL_ADC_PortInit(0);

	// Set ADC external clock
	HAL_SCU_MiscClockConfig(7, ADC_TYPE, CLKSRC_MCLK, 8);

	// Initialize and configure ADC
	ADC_Config.TriggerInfo = DISABLE;
	ADC_Config.ChannelInfo = ENABLE;
	ADC_Config.Dma = DISABLE; // DMA Enable
	ADC_Config.Restart = DISABLE;
	ADC_Config.Mode = ADC_MODE_SINGLE;
	ADC_Config.TriggerSelection = ADC_TRIGGER_SEL_OLNY_SOFT;
	ADC_Config.SamplingTime = 1;
	ADC_Config.SequenceCounter = ADC_SEQ_NUM_1;
	ADC_Config.ExternalClock = ENABLE;
	ADC_Config.InternalClockDivider = 1;
	HAL_ADC_Init(ADC, &ADC_Config);

	// DMA Configuration to use DAC Interrupt
	DMA_Config.Peripheral = DMAC_PERIPHERAL_ADC;					// Selected DMA ADC
	DMA_Config.Width = DMAC_WIDTH_HALF_WORD;						// Half-Word byte transfer (16-bit)
	DMA_Config.Direction = DMAC_DIRECTION_RX;						// Peripheral to designated address
	DMA_Config.PeripheralAddress = (uint32_t)(&(ADC->DDR));			// ADC Data register
	HAL_DMAC_Init(DMA_TEST_CH, &DMA_Config);						// DMA Channel Initization
	HAL_DMAC_SetMemoryAddress(DMA_TEST_CH, DMA_ADDR_PERI);          // Setting Memory Address (DAC->DR)
	HAL_DMAC_SetTransferCount(DMA_TEST_CH, DMA_DATA_SIZE);			// Setting DMA Transfer Count Value

	// DMA Interrupt, EOS Interrupt Enable
	HAL_ADC_ConfigureInterrupt(ADC, (ADC_INTERRUPT_Type)(ADC_INTERRUPT_SINGLE | ADC_INTERRUPT_DMA), ENABLE);

	// Enable SysTick and ADC interrupt
	SysTick_Config(SystemCoreClock / 1000);

	NVIC_SetPriority(ADC_IRQn, 3);
	NVIC_EnableIRQ(ADC_IRQn);

	__enable_irq();

	cputs("ADC->DDR values are wrote at ADC->DR via DMA\r\n");

	HAL_ADC_Stop(ADC);
	HAL_DAC_Stop(DAC);

	// Start ADC test
	DMA_Done = FALSE;

	while(1) {
		HAL_ADC_ConfigureOneSequence(ADC, ADC_SEQUENCE_0, (ADC_CHANNEL_Type)(ADC_CHANNEL_0));
		DoneFlag = FALSE;

		HAL_ADC_Start(ADC);
		HAL_ADC_DmaEnable(ADC);	// ADC DMA Enable should be set after ADC_CR<EN> is set to 1.

		HAL_DAC_Start(DAC);	// DAC Start

		while(DoneFlag == FALSE);
		ADC_Data = HAL_ADC_GetConversionData(ADC, ADC_DATA_SEL_LAST);	// Get A/D Conversion Data
        cprintf("ADC Value: %04X\r\n", ADC_Data);   
        
		if (DMA_Done == TRUE) {										    // Check DMA Flag
			HAL_DMAC_Stop(DMA_TEST_CH);									// If DMA is done, Stop DMA operation
			DMA_Done = FALSE;
		}

		HAL_DMAC_Start(DMA_TEST_CH);									// DMA Operation Start
        DAC_Data = HAL_DAC_GetData(DAC);                                // Get DAC Data
        cprintf("DAC Value: %04X\r\n", DAC_Data);
        
		// 1 second delay
		MilliSec = 1000;
		while(MilliSec);
	}
}

/* --------------------------------- End Of File ------------------------------ */
