/*!
 * @file        system_bmp561.c
 *
 * @brief       CMSIS Cortex-M0+ Device Peripheral Access Layer System Source File
 *
 * @version     V1.0.0
 *
 * @date        2025-01-01
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

#include "bmp561.h"

/** @addtogroup CMSIS
  @{
*/

/** @addtogroup BMP561_System
  @{
*/

/** @defgroup System_Variables
  @{
*/

uint32_t SystemCoreClock = SYSCLK_VALUE;

/**@} end of group System_Variables */

/** @defgroup System_Functions
  @{
*/

static void SystemClockConfig(void);

/*!
 * @brief       Setup the microcontroller system
 *
 * @param       None
 *
 * @retval      None
 *
 */
void SystemInit (void)
{
    /* Unlock SYSCTRL */
    SYSCTRL->KEY_B.LOCKKEY = 0x87E4;

    SystemClockConfig();

    /* Lock SYSCTRL */
    SYSCTRL->KEY_B.KEYST = 1;

    SCB->VTOR = 0x08000000;
}

/*!
 * @brief       Update SystemCoreClock variable according to Clock Register Values
 *              The SystemCoreClock variable contains the system clock
 *
 * @param       None
 *
 * @retval      None
 *
 */
void SystemCoreClockUpdate (void)
{
    uint32_t sysClockDiv;
    uint32_t hsiClockDiv;

    sysClockDiv = SYSCTRL->SCCR_B.SYSCLKDIV;
    hsiClockDiv = SYSCTRL->HSICLKCR_B.HSICLKDIV;
    SystemCoreClock = SYSCLK_VALUE / (1 << sysClockDiv) /  (1 << hsiClockDiv);
}

/*!
 * @brief       Configures the System clock frequency and APB clock division
 *
 * @param       None
 *
 * @retval      None
 *
 */
static void SystemClockConfig(void)
{
    /* Configure system clock division */
    SYSCTRL->SCCR_B.SYSCLKDIV = 0;
    /* Configure HSI clock division */
    SYSCTRL->HSICLKCR_B.HSICLKDIV = 0;
    /* Configure APB clock division */
    SYSCTRL->SCCR_B.APBCLKDIV = 0;
}

/**@} end of group System_Functions */
/**@} end of group BMP561_System */
/**@} end of group CMSIS */
