/*
 ***********************************************************************************************************************
 *
 * Copyright (c) 2015, Infineon Technologies AG
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,are permitted provided that the
 * following conditions are met:
 *
 *   Redistributions of source code must retain the above copyright notice, this list of conditions and the  following
 *   disclaimer.
 *
 *   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *   following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 *   Neither the name of the copyright holders nor the names of its contributors may be used to endorse or promote
 *   products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE  FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT  OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** BG           Blandine Guillot                                              **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2020-02-27, BG:   Initial version of revision history              **
*******************************************************************************/
 
/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** Timer2: Measurement of the PWM (duty cycle/frequency)                      **
********************************************************************************
** Two boards are necessary for this example:                                 **
**  - 1st board: flash this example -> master board                           **
**  - 2nd board: flash the CCU6_PWM_Generation example -> slave board         **
** Connect P1.2 of the master board with one of the PWM output pins of the    **
** slave board.                                                               **
**                                                                            **
** Measuring the low part of the PWM:                                         **
** Timer2 starts counting with a falling edge on P1.2                         **
** Timer2 captures on the rising edge on P1.2 (PWM low phase, capDC)          **
**                                                                            **
** Measuring the period of the PWM:                                           **
** Timer2 still runs from previous falling edge                               **
** Timer2 captures on the falling edge on P1.2 (PWM period, capPer)           **
**                                                                            **
** Always one period is skipped in order to reprogram the edge detection of   **
** the Timer2.                                                                **
**                                                                            **
** Max. PWM period is 1.6ms (625Hz)                                           **
** @ 1kHz PWM frequency (1ms) the min. recognizable DC is 0.3%, max. 99.8%    **
********************************************************************************
**         start          capture   capture                capture            **
**         Timer2         Timer2    Timer2                 Timer2             **
**       -----                --------              ---------                 **
** P1.2       |              |        |            |         |                **
**            :--------------:\       :------------          :------...       **
**            :    capDC     : \      :\                     :    capDC       **
**            :<------------>:  \     : \                    :<-----...       **
**            :                  \    :  \                   :                **
**            :     PWM_Capture() ISR :   \                  :                **
**           (captured on rising edge):    \                 :                **
**            :                       :     \                :                **
**            :    capPer             :      \               :    capPer      **
**            :<--------------------->:       \              :<-----...       **
**                                             \                              **
**                                      PWM_Capture() ISR                     **
**                            (captured on  falling edge)                     **
*******************************************************************************/

#include "tle_device.h"
#include "eval_board.h"

static uint16 capPer, capDC;
static uint16 bDC_Per;

int main(void)
{
  /*****************************************************************************
  ** Initialization of the hardware modules based on the configuration set    **
  ** in Config Wizard                                                         **
  *****************************************************************************/
  TLE_Init();
  
  bDC_Per = 0;

  for (;;)
  {
    (void)WDT1_Service();    
  }
}


/* Callback function for Timer2 */
void PWM_Capture(void)
{
  /* Timer2 in run mode */
  if (TIMER2->T2CON.bit.TR2 == 1u)
  {
    if (bDC_Per == 0u)
    {
      /* Get the duty cycle */
      capDC = TIMER2_Get_Capture();
      /* Next capture on falling edge */
      TIMER2->T2MOD.bit.EDGESEL = 0u;

      bDC_Per = 1u;
    }
    else
    {
      /* Get the PWM period */
      capPer = TIMER2_Get_Capture();
      /* Next capture on rising edge */
      TIMER2->T2MOD.bit.EDGESEL = 1u;

      bDC_Per = 0u;
      /* Stop Timer2 */
      TIMER2_Stop();
      /* Reset Timer2 */
      TIMER2_Clear_Count();
    }
  }
}
