/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** BG           Blandine Guillot                                              **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2020-02-27, BG:   Initial version of revision history              **
** V1.0.1: 2021-07-19, BG:   Removed warnings for ARMCC v6 compiler           **
** V1.0.2: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/
 
/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** BDRV: Run a Brush DC motor with 25% duty cycle and diagnostic              **
********************************************************************************
** Timer12 (CCU6) is running with a period of 50 us                           **
** Channels 0 and 1 are connected to BDRV and configured with 25% duty cycle  **
** Before the motor is turned on, an Openload detection is performed:         **
**  - if no motor is connected, LED1 (P0.1) is turned on                      **
**  - if a motor is connected, it must be spinning                            **
** The Overcurrent diagnostic is enabled. In case of an overcurrent condition **
** or a short of the motor to either VBAT or GND, the following LEDs provide  **
** information on where it was detected:                                      **
**  - LED6 = LS1                                                              **
**  - LED5 = HS1                                                              **
**  - LED4 = LS2                                                              **
**  - LED3 = HS2                                                              **
*******************************************************************************/

#include "tle_device.h"
#include "eval_board.h"

int main(void)
{
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();

  /* Check openload at the motor contacts */
  if (BDRV_Diag_OpenLoad() == true)
  {
    /* No motor detected */
    PORT_ChangePin(LED1, PORT_ACTION_SET);
  }
  else
  {
    /* Motor detected */

    /* Configure bridge to run PWM on Phase 2 - forward */
    //BDRV_Set_Bridge(Ch_En, Ch_On, Ch_PWM, Ch_PWM);

    /* Configure bridge to run PWM on Phase 1 - backward */
    BDRV_Set_Bridge(Ch_PWM, Ch_PWM, Ch_En, Ch_On);

    /* Start PWM generation */
    CCU6_StartTmr_T12();
  }

  for (;;)
  {
    /* Main watchdog service */
    (void)WDT1_Service();
  }
}


#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6000000)
  #pragma clang diagnostic push
  #pragma clang diagnostic ignored "-Wmissing-prototypes"
#endif
void Bdrv_Diagnose(void)
{
  /* Clear all LEDs used for diagnosis */
  PORT_ChangePin(LED6, PORT_ACTION_CLEAR);
  PORT_ChangePin(LED5, PORT_ACTION_CLEAR);
  PORT_ChangePin(LED3, PORT_ACTION_CLEAR);
  PORT_ChangePin(LED4, PORT_ACTION_CLEAR);

  if (BDRV->CTRL1.bit.LS1_OC_STS == 1)
  {
    /* Overcurrent detected on Low Side 1 */
    /* LED6 turned on */
    PORT_ChangePin(LED6, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.LS1_OC_ICLR = 1;
  }
  if (BDRV->CTRL1.bit.HS1_OC_STS == 1)
  {
    /* Overcurrent detected on High Side 1 */
    /* LED5 turned on */
    PORT_ChangePin(LED5, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.HS1_OC_ICLR = 1;
  }
  if (BDRV->CTRL1.bit.LS2_OC_STS == 1)
  {
    /* Overcurrent detected on Low Side 2 */
    /* LED4 turned on */
    PORT_ChangePin(LED4, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.LS2_OC_ICLR = 1;
  }
  if (BDRV->CTRL1.bit.HS2_OC_STS == 1)
  {
    /* Overcurrent detected on High Side 2 */
    /* LED3 turned on */
    PORT_ChangePin(LED3, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.HS2_OC_ICLR = 1;
  }

  if ((BDRV->CTRL1.bit.LS1_DS_STS == 1) && (BDRV->CTRL1.bit.LS1_DCS_EN == 1))
  {
    /* Short detected on Low Side 1 */
    /* LED6 turned on */
    PORT_ChangePin(LED6, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.LS1_DS_ICLR = 1;
  }
  if ((BDRV->CTRL1.bit.HS1_DS_STS == 1) && (BDRV->CTRL1.bit.HS1_DCS_EN == 1))
  {
    /* Short detected on High Side 1 */
    /* LED5 turned on */
    PORT_ChangePin(LED5, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.HS1_DS_ICLR = 1;
  }
  if ((BDRV->CTRL1.bit.LS2_DS_STS == 1) && (BDRV->CTRL1.bit.LS2_DCS_EN == 1))
  {
    /* Short detected on Low Side 2 */
    /* LED4 turned on */
    PORT_ChangePin(LED4, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.LS2_DS_ICLR = 1;
  }
  if ((BDRV->CTRL1.bit.HS2_DS_STS == 1) && (BDRV->CTRL1.bit.HS2_DCS_EN == 1))
  {
    /* Short detected on High Side 2 */
    /* LED3 turned on */
    PORT_ChangePin(LED3, PORT_ACTION_SET);
    /* Clear status bit */
    SCUPM->BDRV_ISCLR.bit.HS2_DS_ICLR = 1;
  }
  
  WDT1_Service();
}
#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6000000)
  #pragma clang diagnostic pop
#endif

