/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** JM           Jahangir Mansoori                                             **
** BG           Blandine Guillot                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2016-07-01, JM:   Initial version                                  **
** V1.0.1: 2020-11-05, JO:   Updated description of the example               **
** V1.0.2: 2021-07-19, BG:   Removed warnings for ARMCC v6 compiler           **
** V1.1.1: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** GPT12E: Input Capture                                                      **
********************************************************************************
** Timer2 toggles P0.2 for each overflow (callback TIMER2_ISR1) after         **
** 26.66ms.                                                                   **
** The core timer 3 of GPT12E is set up as a timer with 310 kHz and it        **
** toggles P1.2 for each overflow (callback T3_ISR_Overflow).                 **
** The auxiliary timer T2 of GPT12E is configured to capture on either edge   **
** of P1.4 (callback T2_CAP_ISR).                                             **
** Use a jumper wire to connect P0.2 with P1.4 so that the timer T2 of GPT12E **
** can count between rising-falling pulses.                                   **
** The variable final_counts holds the captured value in ticks.               **
*******************************************************************************/

#include "tle_device.h"

void TIMER2_ISR1(void);
void T2_CAP_ISR(void);
void T3_ISR_Overflow(void);

__USED static uint32 flag_overflow;
__USED static uint32 counts, previous_counts;
__USED static uint32 final_counts;


int main(void)
{
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  TIMER2_Start();
  GPT12E_T3_Start();

  for (;;)
  {
    (void)WDT1_Service();
  }
}


/* Timer2 toggles P0.2 for each overflow after 26.66ms */
void TIMER2_ISR1(void)
{
  PORT_ChangePin(0x02, PORT_ACTION_TOGGLE);
}


/* Capture ISR, Rising-Falling interrupt at P1.4 */
void T2_CAP_ISR(void)
{
  counts = GPT12E_T2_Value_Get();

  if (flag_overflow == 0)
  {
    final_counts = counts - previous_counts;
  }
  else
  {
    final_counts = (uint32)((0xffff - previous_counts) + counts);
    flag_overflow--;
  }

  previous_counts = counts;
}


/* Core timer 3 of GPT12E toggles P1.2 for each overflow */
void T3_ISR_Overflow(void)
{
  flag_overflow++;

  if (flag_overflow > 2)
  {
    flag_overflow = 0;
    counts = 0;
    final_counts = 0;
  }
}
