/*****************************************************************************\
*                                                                             *
*                _  _               ___             _                         *
*               (_)| |__   _ __    / _ \ _ __ ___  | |__    /\  /\            *
*               | || '_ \ | '__|  / /_\/| '_ ` _ \ | '_ \  / /_/ /            *
*               | || | | || |    / /_\\ | | | | | || |_) |/ __  /             *
*               |_||_| |_||_|    \____/ |_| |_| |_||_.__/ \/ /_/              *
*                                                                             *
*  ihr GmbH                                                                   *
*  Airport Boulevard B210                                                     *
*  77836 Rheinmnster - Germany                                               *
*  http://www.ihr.de                                                          *
*  Phone +49(0) 7229-18475-0                                                  *
*  Fax   +49(0) 7229-18475-11                                                 *
*                                                                             *
*******************************************************************************
*                                                                             *
*                                                                             *
* (c) Alle Rechte bei IHR GmbH, auch fuer den Fall von Schutzrechts-          *
* anmeldungen. Jede Verfuegungsbefugnis, wie Kopier- und Weitergaberecht      *
* bei uns.                                                                    *
*                                                                             *
* (c) All rights reserved by IHR GmbH including the right to file             *
* industrial property rights. IHR GmbH retains the sole power of              *
* disposition such as reproduction or distribution.                           *
*                                                                             *
*                                                                             *
********************   Workfile:      lin_driver_api.c      *******************
*                                                                             *
*  PROJECT-DESCRIPTION:  LIN Driver Protocol Layer                            *
*                                                                             *
*  FILE-DESCRIPTION:     All routines for access from application to          *
*                       lin driver and callback function from lin driver      *
*                             to application                                  *
*                                                                             *
*******************************************************************************
*       Revision:       1.0                                                   *
*       Responsible:    B. Roegl                                              *
*       Creation date:  18/02/2014                                            *
*       Last Modtime:   18/02/2014                                            *
*                                                                             *
*  HISTORY:                                                                   *
*                                                                             *
*    Rev 1.0   18/02/2014 by H. Spinner                                       *
*         - New Demo driver for TLE986x/7x                                    *
*                                                                             *
\*****************************************************************************/
/**
@file    lin_driver_api.c
@brief   Application Program Interface for the LIN Driver access
*/

/* ===========================================================================
 *  Header files
 * ==========================================================================*/

#include "genLinConfig.h"
#include "lin_driver_api.h"

#include "lin_slave_task.h"
#include "lin_master_task.h"

/* ===========================================================================
 *  Global Variables
 * ==========================================================================*/

/* ===========================================================================
 *  Functions
 * ==========================================================================*/

/* ---------------------------------------------------------------------------
 *  l_bool l_sys_init (void)
 * --------------------------------------------------------------------------*/
/**
   @brief  Initialise LIN driver system.
   @pre    MCU started
   @param  void
   @retval void
*/
l_bool l_sys_init (void)
{
   lin_main_init();
   return 0u;
}

#ifdef LIN_MASTER
/* ---------------------------------------------------------------------------
 *  l_u8 l_sch_tick (void)
 * --------------------------------------------------------------------------*/
/**
  @brief    Periodic check of LIN scheduletable and to required action
   
    The function follows a schedule. When a frame becomes due,
    its transmission is initiated. When the end of the current schedule
    is reached, l_sch_tick starts again at the beginning of the schedule.
    The l_sch_tick must be called periodically and individually for each
    interface within the node. The period is the time base set in the LDF.
    The period of the l_sch_tick call effectively sets the time base tick.
    Therefore it is essential that the time base period is uphold with
    minimum jitter. The call to l_sch_tick will not only start the
    transition of the next frame due, it will also update the signal 
    values for those signals received since the previous call to l_sch_tick.
  @pre     LIN driver initialised
  @param   void
  @return  void
*/
l_u8 l_sch_tick (void)
{
   l_u8  temp_lin_proto;
   l_u8  temp_frm_length;
   l_u8  retval = 0u;

   if(!g_lin_zero_schedule)
   {
      /* check if schedule table at beginning to switch to other scheduletable
       * After sending an Goto Sleep command, do not switch to a new scheduletable
       * during diagnosis do not change the table */
      if((!g_lin_zero_schedule) && ((g_ScheduleIdx != g_nextScheduleIdx) || (g_nextScheduleFrmIdx != 0xFFu)))
      {
         g_ScheduleIdx =  g_nextScheduleIdx;
         g_activeScheduleIdx = g_ScheduleIdx;
         g_ScheduleFrmIdx = g_nextScheduleFrmIdx;
         g_nextScheduleFrmIdx = 0xFFu;
         if ((*scheduleList[g_activeScheduleIdx]) == 0)
         {
            g_lin_zero_schedule = L_SET;
         }
      }
      /* check if its time to send next frame header */
      if (!g_lin_zero_schedule)
      {
         if(g_lin_sched_data.lin_current_delay == 0u)
         {
            if(((*scheduleList[g_activeScheduleIdx])[g_ScheduleFrmIdx].frame_type & 0x0080u) == 0x0080u)
            {
               temp_lin_proto = LIN_2_X;
            }
            else
            {
               temp_lin_proto = LIN_1_3;
            }
            temp_frm_length = (l_u8)(((*scheduleList[g_activeScheduleIdx])[g_ScheduleFrmIdx].frame_type & 0xFF00u) >> 8u);

            /* Finish previous frame via timeout procedure to ensure every state is correctly reset upon starting new frame header */
            lin_slave_state_g = idle;

            /*  Start next Frame in Schedule */
            lin_master_task_tx_header(temp_lin_proto,(l_u8)(*scheduleList[g_activeScheduleIdx])[g_ScheduleFrmIdx].pid,temp_frm_length);

            /*  Get the delay until the next Frame */
            g_lin_sched_data.lin_current_delay = (*scheduleList[g_activeScheduleIdx])[g_ScheduleFrmIdx].frame_delay;


            /*  Prepare for next Frame transmition */
            /* get next frame in schedule */
            g_ScheduleFrmIdx ++;
            /* cycle in schedule */
            g_ScheduleFrmIdx %= (schedTblSizeList[g_ScheduleIdx]);

            /* reduce lin_current_delay with main_cyclic_looptime(ms) in 100s */
            g_lin_sched_data.lin_current_delay -=(10u * (LIN_TASK_CYCLE_MS));
         }
         else /* no frame valid this time */
         {
            /* reduce lin_current_delay with main_cyclic_looptime(ms) in 100s */
            if (g_lin_sched_data.lin_current_delay >= (10u * (LIN_TASK_CYCLE_MS)))
            {
               g_lin_sched_data.lin_current_delay -=(10u * (LIN_TASK_CYCLE_MS));
            }
            else
            {
               g_lin_sched_data.lin_current_delay = 0u;
            }
            if(g_lin_sched_data.lin_current_delay == 0u)
            {
               retval = g_ScheduleFrmIdx + 1u;
            }
            else
            {
               retval = 0u;
            }
         }
      } /* end if active schedule != 0 */
   }
   return retval;
}
#endif /* end #ifdef LIN_MASTER */

#ifdef LIN_MASTER
/* ---------------------------------------------------------------------------
 *  void l_sch_set (l_u16 schedule, l_u8 entry)
 * --------------------------------------------------------------------------*/
/**
  @brief   The function sets a new schedule in the schedule management
  @pre     LIN driver initialised
  @param   schedule  The new schedule table
  @param   entry  The next frame in new schedule table which had to be executed
  @return  void
*/
void l_sch_set (l_u16 schedule, l_u8 entry)
{
   if(schedule < NumberOfSchedTables)
   {
      g_nextScheduleIdx = schedule;
      g_nextScheduleFrmIdx = entry;
      g_lin_zero_schedule = L_RESET;
   }
   else
   {
      g_lin_zero_schedule = L_SET;
   }
}

void l_sch_change_pid (l_u8 old_pid, l_u8 new_pid)
{
   l_u8 loc_j, loc_k;
   for (loc_j = 0u; loc_j < NumberOfSchedTables; loc_j++)
   {
      for (loc_k = 0u; loc_k < schedTblSizeList[loc_j]; loc_k ++)
      {
         if ((*scheduleList[loc_j])[loc_k].pid == old_pid)
         {
            (*scheduleList[loc_j])[loc_k].pid = new_pid;
         } 
/*         if ((*l_ptr_scheduleList)[loc_k][0] == old_pid)
         {
            (*l_ptr_scheduleList)[loc_k][0] = new_pid;
         } */
      }
   }
}

#endif /* end #ifdef LIN_MASTER */

/* ---------------------------------------------------------------------------
 *  l_bool l_ifc_init(void)
 * --------------------------------------------------------------------------*/
/**
   @brief  Initialize LIN interface.
   @pre    LIN driver initialized
   @param  void
   @retval State of initialization
*/

l_bool l_ifc_init(void)
{
   return 0u;
}

#ifndef LIN_PROTOCOL_VERSION_1_3
/* ---------------------------------------------------------------------------
 *  l_u16 l_ifc_read_status(void)
 * --------------------------------------------------------------------------*/
/**
   @brief  Get status information from LIN driver.
   @pre    LIN driver initialized
   @param  void
   @retval LIN_status word        MSB contains last PID information
                                  LSB contains status flags
   @see    LIN 2.0 and LIN 2.1 specification
*/
l_u16 l_ifc_read_status(void)
{
   l_u16 loc_temp;

   loc_temp = g_lin_status_word.reg;
   g_lin_status_word.reg = 0u;
   return loc_temp;
}
#endif /* ifndef LIN_PROTOCOL_VERSION_1_3 */

