/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** KC           Kay Claussen                                                  **
** BG           Blandine Guillot                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2016-10-13, KC:   Initial version                                  **
** V1.0.1: 2020-02-27, BG:   Updated abstract                                 **
** V1.0.2: 2021-07-19, BG:   Removed warnings for ARMCC v6 compiler           **
** V1.2.1: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** LIN: Autobaudrate detection                                                **
********************************************************************************
** Timer 2 is configured to capture the LIN baudrate. Read out of baudrate    **
** is done after synch field detection in LIN_End_Of_Synch_Handler().         **
** Device generates interrupt on reception of                                 **
** - LIN Synch Field -> Toggles P0.1                                          **
** - LIN Synch Field errors -> Toggles P0.2                                   **
** - T2 Interrupt on falling edge of LIN signal-> Toggles P0.3                **
** - UART byte reception -> Toggles P1.0                                      **
** Variable LIN_baudrate_ticks keeps the captured time of 8 bits:             **
** With TLE9869QXA20: LIN_baudrate_ticks ~ 10015 ticks at baudrate 19200      **
**                    f_sys = 24 MHz => T_sys = 1/24000000 ~ 42 ns/tick       **
**                    => 10015[tick] x 42[ns/tick] ~ 417292 ns for 8 bits     **
**                    => 417292[ns/8bits] / 8[bit] ~ 52.16 us for 1 bit       **
**                    => 1[bit] / 0.00005216[s] ~ 19171 bit/s = 19171 baud    **
*******************************************************************************/

#include "tle_device.h"

#define ERRSYNC_EOFSYNC_BRK 0x38 /* mask for lin break, synch and synch error */

__USED static uint16 LIN_baudrate_ticks;

int main(void)
{
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();

  for (;;)
  {
    (void)WDT1_Service();
  }
}

/* ISR for LIN End of Synch Field detection */
void LIN_End_Of_Synch_Handler(void)
{
  /* Toggle P0.1 */
  PORT_P01_Output_Toggle_Set();

  /* If break field AND synch field AND no sync error appeared then ... */
  if ((SCU->LINST.reg & ERRSYNC_EOFSYNC_BRK) == 0x18)
  {
    /* ... capture timer value */
    LIN_baudrate_ticks = TIMER2_Get_Capture();
  }
}

/* ISR for LIN synch filed error detection */
void LIN_Synch_Error_Handler(void)
{
  /* Toggle P0.2 */
  PORT_P02_Output_Toggle_Set();
}

/* ISR for T2 */
void TIMER2_Handler(void)
{
  /* Toggle P0.3 */
  PORT_P03_Output_Toggle_Set();
  TIMER2_Stop();
  TIMER2_Clear_Count();
  /* Clear  LIN field indicators */
  SCU->LINSCLR.reg = ERRSYNC_EOFSYNC_BRK;
  /* Disable external T2 interrupt */
  TIMER2_External_Int_Dis();
}

/* ISR for UART Receive */
void UART_Receive_Handler(void)
{
  /* Toggle P1.0 */
  PORT_P10_Output_Toggle_Set();
  /* In a LIN driver this part of the code should be located after reception of the last LIN byte or after the occurrence of an error on LIN. */
  /* This will reduce the number of T2 interrupts. */
  /* Clear pending T2 interrupt */
  TIMER2->T2ICLR.reg = 0x40;
  /* Enable external T2 interrupt */
  TIMER2_External_Int_En();
}
