/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** DM           Daniel Mysliwitz                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2015-09-22, DM:   Initial version                                  **
** V1.0.4: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** Flash: Data flash error handling                                           **
********************************************************************************
** In this example, we try to write page 0 of the Data flash sector.          **
** Possible errors are handled during startup.                                **
*******************************************************************************/

#include "tle_device.h"
#include "eval_board.h"
#include <string.h>
#include "nvm_func.h"

#define _User_Page_Cnt_ (3u)
#define _pDataFlashPage0_ DataFlashStart+(0 * 0x80)
#define _pDataFlashPage1_ DataFlashStart+(1 * 0x80)
#define _pDataFlashPage2_ DataFlashStart+(2 * 0x80)

const uint8 PageInitData[3] = {0xAA, 0x33, 0xCC};
__USED extern volatile bool bDataFlashEnabled;
__USED extern uint8 TargetData[FlashPageSize];
__USED volatile uint8 PageCnt;

int main(void)
{
  uint8 data;
  static uint8 *pDataFlashAddr;
  uint8 lMEMSTAT;
  TlocProgramPage lprog_res;
  TlocEraseSector ler_res;
  bool bInitData;
  bDataFlashEnabled = false;
  bInitData = false;
  /* Dummy assignment to remove compiler warning */
  data = data;
  
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  /* Save MEMSTAT in a local variable for later evaluation */
  lMEMSTAT = SCU->MEMSTAT.reg;
  /* Clear MEMSTAT as it acts as input into the NVM function for Emergency Operation */
  SCU->MEMSTAT.reg = 0u;

  /* Check if MapRAM Initialization has failed */
  if (SCU->SYS_STRTUP_STS.bit.MRAMINITSTS == 1u)
  {
    /* !!!Data Flash consistency not given anymore!!! */
    /* Check retry counter */
    if (PMU->GPUDATA00.reg > 0u)
    {
      /* This is already the second try */
      /* Data flash error seems not to be fixable by the Service Algorithm */
      /* SECTOR_ERASE will be executed */
      ler_res = locEraseSector(DataFlashStart);

      if (ler_res.bit.GlobFail == 1u)
      {
        /* Signal to the application that Data Flash is not ok */
        bDataFlashEnabled = false;
      }
      else
      {
        /* Reset retry counter */
        PMU->GPUDATA00.reg = 0u;
        /* Signal to the application that Data Flash is ok */
        bDataFlashEnabled = true;
        /* Since the data flash was erased, initialize the user pages */
        bInitData = true;
      }
    }
    else
    {
      /* Increment retry counter */
      PMU->GPUDATA00.reg++;
      /* Force WDT1 reset in order to restart device */
      SCUPM->WDT1_TRIG.reg = 1u;
      SCUPM->WDT1_TRIG.reg = 1u;
      /* Device performs a reset */
    }
  }
  else
  {
    /* Data flash is consistent */
    /* Signal to the application that Data Flash is ok */
    bDataFlashEnabled = true;
  }

  /* Check Service Algorithm return value (MEMSTAT) */
  /* or if data flash pages need to be initialized again */
  if ((lMEMSTAT != 0u) || (bInitData == true))
  {
    /* If SECTORINFO is not 0, then the Service Algorithm was active */
    /* Potentally it has fixed something in the data flash. */
    /* The user application should now check whether all expected data flash pages are still present */
    /* Clear MAPNMI flag  */
    SCU->NMICLR.bit.NMIMAPC = 1u;
    /* Let's start with checking the first user page */
    PageCnt = 0;

    while ((PageCnt < _User_Page_Cnt_) && (bDataFlashEnabled == true))
    {
      /* Preload the TargetData array with the initialization data */
      memset(&TargetData, PageInitData[PageCnt], FlashPageSize);
      /* Assign a pointer to the data flash page start address */
      /* Make a dummy ready into user page 0 */
      /* If page is not present, an MAPNMI will be raised */
      pDataFlashAddr = (uint8 *)(DataFlashStart + (PageCnt * FlashPageSize));
      data = *pDataFlashAddr;
      /* Dummy operation to prevent compiler from optimizing away */
      /* data = *pDataFlashAddr read operation */
      data = data + 1;
      /* Check next page */
      PageCnt++;
    }
  }

  for (;;)
  {
    (void)WDT1_Service();

    /* Check level of P1.0: if P1.0 is high, page 0 will be programmed */
    /* and the device will be sent into Sleep Mode. */
    /* The device wakes up automatically after approx. 2 sec (cyclic wake) */
    if (PORT_ReadPin(0x10) == 1u)
    {
      /* Example data flash operation: */
      /* Byte0 of Page0 is read, inverted and a set of 128 bytes with */
      /* the inverted data is being prepare and stored again in Page0. */
      /* The data in Page0 gets inverted with every programming. */
      pDataFlashAddr = (uint8 *)(_pDataFlashPage0_);
      data = *pDataFlashAddr;
      memset(&TargetData, data ^ (uint8)0xFF, FlashPageSize);
      lprog_res = locProgramPage(_pDataFlashPage0_, TargetData);

      if (lprog_res.bit.GlobFail == 1u)
      {
        /* Page programming failed even with local retry */
        /* If the supply was too low, then the programming can be tried again. */
        /* If the VS is stable again then the programming should work. */
        /* If the programming failed for other reasons, then either the fail reasons are*/
        /* covered already inside the function locProgramPage() or the usage of the data*/
        /* flash gets temporarly disabled by setting the variable bDataFlashEnabled to 'false'.*/
      }

      /* Send the device into Sleep Mode, with Cyclic Wake enabled  */
      SCU_EnterSleepMode();
    }
  }
}

/* Initialisation of missing data flash pages (optional) */
void NMI_MAP_ISR(void)
{
  TlocEraseSector ler_res;
  TlocProgramPage lprog_res;
  /* Call the local program page function */
  lprog_res = locProgramPage(DataFlashStart + (PageCnt * FlashPageSize),
                             TargetData);

  /* Check if locProgramPage has failed */
  if (lprog_res.bit.GlobFail == 1u)
  {
    /* The retry of the page programming has failed */
    /* Let's check what has failed */
    if (lprog_res.bit.VS2lowFail == 1u)
    {
      /* Even after a retry, the VS was still too low. */
      /* Further NVM operations will be disabled. */
      /* The application at some later point in time should check the VS */
      /* and repeat this programming step if possible. */
      bDataFlashEnabled = false;
    }

    if (lprog_res.bit.RetryFail == 1u)
    {
      /* The desired numbers of programming retries have been performed, */
      /* but sill failures did remain. */
      /* The data flash does not seem to be in a good condition anymore. */
      /* A complete sector erase could be tried. Potentially the data flash is in end-of-life condition. */
      ler_res = locEraseSector(DataFlashStart);

      if (ler_res.bit.GlobFail == 1u)
      {
        /* Signal to the application that Data Flash is not ok */
        bDataFlashEnabled = false;
      }
      else
      {
        /* Reset retry counter */
        PMU->GPUDATA00.reg = 0u;
        /* Signal to the application that Data Flash is ok */
        bDataFlashEnabled = true;
        /* Try again to program this page */
        PageCnt--;
      }
    }
  }
  else
  {
    /* locProgramPage passed, nothing to be done here, continue with next page */
  }
}
