/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

#include "tle_device.h"
#include "nvm_func.h"
#include "scu_defines.h"

/*******************************************************************************
** VS_Min, needs to be calculated based on the required time for the device   **
** to stay above 3V in order to finish any NVM operation                      **
** The current consumption at VS and the voltage drop on VS over time         **
**                                                                            **
**     I * t   30mA * 16ms                                                    **
** C = ----- = ----------- = 80uF                                             **
**       U      (9V - 3V)                                                     **
**                                                                            **
** min. 80uF at VS would be required to fulfill the conditions of providing   **
** enough charge at VS to keep the device running above 3V for 16ms           **
**                                                                            **
*******************************************************************************/
#define _VS_Min_mV_ (9000)

/*******************************************************************************
**                         Global Variable Definitions                        **
*******************************************************************************/
/* allocate one flash page in RAM */
uint8 TargetData[FlashPageSize];
volatile bool bDataFlashEnabled;

/*******************************************************************************
**                         Global Function Definitions                        **
*******************************************************************************/
/** \brief Erases a flash sector with checks.
 *
 * \param sector address
 * \return sector erase result
 *
 * \ingroup nvm_func
 */
TlocEraseSector locEraseSector(uint32 sector_address)
{
#define _lMax_Retry_ (2u)
  TlocEraseSector res;
  TUser_Erase_Sector es_res;
  uint32 lRetry;
  lRetry = 0u;

  /* check if ADC2.VS >= 9V                                              **
  ** NVM operations are only safe to be executed if there is sufficient  **
  ** supply voltage available to finish the NVM operation just in case   **
  ** a sudden power loss occurs right in the middle of the NVM operation */
  do
  {
    res.reg = 0u;

    if (ADC2_VS_Result_mV() >= _VS_Min_mV_)
    {
      /* trigger WDT1 Short-Open-Window                                  */
      WDT1_SOW_Service((uint8)1);
      /* erase data flash sector */
      es_res.reg = USER_ERASE_SECTOR(sector_address);
      /* trigger regular WDT1, close SOW                                 */
      (void)WDT1_Service();

      if (es_res.bit.EraseFail == 1u)
      {
        /* set the global error in the return value                      */
        res.bit.GlobFail = 1u;

        /* check for further details of the error                        */
        if (es_res.bit.ExecFail == 1u)
        {
          /* this error happens if the function gets called but could    **
          ** not be executed, reasons are:                               **
          **                                                             **
          ** --------------------------------------------------------    **
          ** Error Reason             |  Corrective Action               **
          ** --------------------------------------------------------    **
          ** Flash protected          | remove protection temporarly     **
          **                          | and try again                    **
          ** --------------------------------------------------------    **
          ** address out of range     | check the input parameter        **
          **                          | "addr", check the code           **
          ** --------------------------------------------------------    */
          USER_DFLASH_WR_PROT_DIS(SCU_DFLASH_WPROT_PW);
          res.bit.ExecFail = 1u;
        }
        else
        {
          /* if it was not an error due to execution trouble, then the   **
          ** sector erase itself has failed                              **
          ** as a consequence the data flash will be disabled for the    **
          ** application.                                                */
          bDataFlashEnabled = false;
        }
      }
    }
    else
    {
      /* VS too low */
      res.bit.VS2lowFail = 1u;
      res.bit.GlobFail = 1u;
    }

    lRetry++;
  }
  while ( (res.bit.GlobFail == 1u) &&
          (lRetry < _lMax_Retry_)  &&
          (bDataFlashEnabled == true) );

  if (lRetry >= _lMax_Retry_)
  {
    res.bit.RetryFail = 1u;
    res.bit.GlobFail = 1u;
  }

  return (res);
}

/** \brief Performs a page programming with checks.
 *
 * \param page address, data buffer
 * \return page programming result
 *
 * \ingroup nvm_func
 */
TlocProgramPage locProgramPage(uint32 addr, uint8 *buf)
{
#define _lMax_Retry_ (2u)
  TUser_Prog prog_res;
  TUser_MAPRAM_Init mri_res;
  TlocProgramPage res;
  uint32 lRetry;
  lRetry = 0u;

  /* check if ADC2.VS >= 9V                                              **
  ** NVM operations are only safe to be executed if there is sufficient  **
  ** supply voltage available to finish the NVM operation just in case   **
  ** a sudden power loss occurs right in the middle of the NVM operation */
  if (bDataFlashEnabled == true)
  {
    do
    {
      res.reg = 0u;

      if (ADC2_VS_Result_mV() >= _VS_Min_mV_)
      {
        prog_res.reg = ProgramPage(addr, buf, 0, 1, 0);

        /* evaluate the result of the ProgramPage function                 */
        if (prog_res.bit.GlobFail == 1u)
        {
          /* there was at least one type of failure during the execution   **
          ** of the ProgramPage function                                   **
          ** let's check what it was in detail                             */
          if (prog_res.bit.ExecFail == 1u)
          {
            /* this error happens if the function gets called but could    **
            ** not be executed, reasons are:                               **
            **                                                             **
            ** --------------------------------------------------------    **
            ** Error Reason             |  Corrective Action               **
            ** --------------------------------------------------------    **
            ** Assembly Buffer not open | execute USER_MAPRAM_INIT         **
            **                          | if pass, then try again          **
            ** --------------------------------------------------------    **
            ** Flash protected          | remove protection temporarly     **
            **                          | and try again                    **
            ** --------------------------------------------------------    **
            ** address out of range     | check the input parameter        **
            **                          | "addr", check the code           **
            ** --------------------------------------------------------    */
            USER_DFLASH_WR_PROT_DIS(SCU_DFLASH_WPROT_PW);
            mri_res = locMapRAM_Init();

            /* check MAPRAM INIT return                                    */
            if (mri_res.bit.GlobFail == 1u)
            {
              /* if MAPRAM INIT still fails after a local retry            **
              ** then disable the DataFlash in the application             */
              bDataFlashEnabled = false;
            }

            res.bit.ExecFail = 1u;
            res.bit.GlobFail = 1u;
          }

          if (prog_res.bit.EmergExit == 1u)
          {
            /* the user requested the NVM operation to be exited due to    **
            ** an Emergency Exit, this is controlled by MEMSTAT.Bit1       **
            ** !! THIS FEATURE IS NOT CONSIDERED HERE IN THIS EXAMPLE !!   */
            res.bit.ExecFail = 1u;
            res.bit.GlobFail = 1u;
          }

          if (prog_res.bit.SparePgFail == 1u)
          {
            /* this failure occurs if no Spare Page could be found anymore **
            ** this can only happen if either the Data Flash is not in a   **
            ** consistent condition, or the data flash is at end-of-life   **
            ** Corrective actions would be to execute USER_MAPRAM_INIT     **
            ** if it runs passes, then try again                           */
            mri_res = locMapRAM_Init();

            /* check MAPRAM INIT return                                    */
            if (mri_res.bit.GlobFail == 1u)
            {
              /* if MAPRAM INIT still fails after a local retry            **
              ** then disable the DataFlash in the application             */
              bDataFlashEnabled = false;
            }

            res.bit.SparePgFail = 1u;
            res.bit.GlobFail = 1u;
          }

          if (prog_res.bit.VerifyFail == 1u)
          {
            /* if the verify fails, due to the fact that the               **
            ** page programming was executed with implicit Retry,          **
            ** the currently selected Spare Page seems not to be in good   **
            ** condition anymore (end-of-life). In principle the data flash**
            ** usage should be disabled now.                               */
            bDataFlashEnabled = false;
            res.bit.VerifyFail = 1u;
            res.bit.GlobFail = 1u;
          }
        }
      }
      else
      {
        /* VS too low */
        res.bit.VS2lowFail = 1u;
        res.bit.GlobFail = 1u;
      }

      lRetry++;
    }
    while ( (res.bit.GlobFail == 1u) &&
            (lRetry < _lMax_Retry_)  &&
            (bDataFlashEnabled == true) );
  }
  else
  {
    res.bit.DFlashDis = 1u;
    res.bit.GlobFail = 1u;
  }

  if (lRetry >= _lMax_Retry_)
  {
    res.bit.RetryFail = 1u;
    res.bit.GlobFail = 1u;
  }

  return (res);
}

/** \brief MapRAM Init with checks.
 *
 * \param none
 * \return MapRAM Init result
 *
 * \ingroup nvm_func
 */
TUser_MAPRAM_Init locMapRAM_Init(void)
{
#define _lMax_Retry_ (2u)
  TUser_MAPRAM_Init res;
  uint32 lRetry;
  lRetry = 0u;

  /* execute USER_MAPRAM_INIT to check consitency of the data       **
  ** flash, a new Spare Page is searched by this as well            */
  do
  {
    /* trigger WDT1 Short-Open-Window                               */
    WDT1_SOW_Service((uint8)1);
    res.reg = USER_MAPRAM_INIT();
    /* trigger regular WDT1, close SOW                              */
    (void)WDT1_Service();

    /* evaluate the return value of the USER_MAPRAM_INIT function   */
    if (res.bit.GlobFail == 1u)
    {
      /* something is wrong with the Data Flash consistency         */
      if (res.bit.ExecFail == 1u)
      {
        /* this is not so bad, the function simply could not be     **
        ** executed because maybe the Assembly Buffer is still open **
        ** Corrective action: call USER_ABORTPROG, this will close  **
        ** the Assembly Buffer, then try it again                   */
        USER_ABORTPROG();
      }

      if ((res.bit.DoubleMapping == 1u) ||
          (res.bit.FaultyPage == 1u) )
      {
        /* that is bad, the MapRAM init function has found some     **
        ** inconsitencies which should be fixed before further      **
        ** Data Flash writeings should be performed                 **
        ** Corrective Action: restart the device, to give the       **
        ** Service Algorithm the chance to repair the sector        */
        /* increment retry counter */
        PMU->GPUDATA00.reg++;
        /* cause WDT1 reset in order to restart device */
        SCUPM->WDT1_TRIG.reg = 1u;
        SCUPM->WDT1_TRIG.reg = 1u;
      }
    }
  }
  while ((res.bit.GlobFail == 1u) && (lRetry < _lMax_Retry_));

  return (res);
}

