/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** BG           Blandine Guillot                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2020-02-27, BG:   Initial version of revision history              **
** V1.1.2: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** Flash: Program one page in data flash                                      **
********************************************************************************
** Reading the first data flash page after page erase leads to an NMI.        **
** Inside the NMI ISR, the first data flash page will be filled with data     **
** consecutive read inside the main application.                              **
** Now it is possible to read from them from the data flash.                  **
*******************************************************************************/

#include "tle_device.h"
#include "eval_board.h"

/* Allocate one flash page in RAM */
__USED static uint8 TargetData[FlashPageSize];
__USED static int NVM_Error;

int main(void)
{
  uint32 i;
  uint8 *p;
  static uint16 checksum, data;

  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  checksum = 0;

  /* Fill up the TargetData array with housenumbers */
  for (i = 0; i < FlashPageSize; i++)
  {
    TargetData[i] = (uint8) i;
    checksum += (uint8) i;
  }

  /* Erase Data Flash sector */
  NVM_Error = USER_ERASE_SECTOR(DataFlashStart);

  /* Check if Sector Erase passed */
  if (NVM_Error == 0u)
  {
    /* Perform a read access to the first Data Flash page */
    /* Given the logical page 0 is not mapped to any physical flash page, a NMI will be issued */
    p = (uint8 *) DataFlashStart;
    /* Read DataFlashStart + Offset (0) */
    data = (uint16) p[0];
    /* NMI ISR gets called, page 0 gets loaded with data */

    /* Check if preloading of data inside the NMI ISR did work */
    if (NVM_Error == 0u)
    {
      /* Read DataFlashStart + Offset (0): you should be able to read data this time */
      data = (uint16) 0;

      for (i = 0; i < FlashPageSize; i++)
      {
        data += (uint16) p[i];
      }

      if (data == checksum)
      {
        /* Data in Data Flash page 0 correct */
        PORT_ChangePin(LED1, PORT_ACTION_SET);
      }
      else
      {
        /* Data in Data Flash page 0 wrong */
        PORT_ChangePin(LED6, PORT_ACTION_SET);
      }
    }
    else
    {
      /* An error happened during Page Write in the NMI ISR */
      PORT_ChangePin(LED7, PORT_ACTION_SET);
    }
  }
  else
  {
    /* An error happened during Erase Sector */
    PORT_ChangePin(LED8, PORT_ACTION_SET);
  }

  for (;;)
  {
    (void)WDT1_Service();
  }
}

void NVM_Handle_PageFault(void)
{
  /* Fill up first page with some data */
  /* The TargetData array will be written to page 0 */
  NVM_Error = ProgramPage(DataFlashStart, TargetData, 0, 0, 0);
}
