/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** BG           Blandine Guillot                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2020-02-27, BG:   Initial version of revision history              **
** V1.1.2: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** Flash: Program a page when the write protection is set                     **
********************************************************************************
** Read the customer id to determine the design step.                         **
** Depending on the design step, the correct method is applied to set the     **
** protection.                                                                **
** A page write is performed in the protected NVM region. This operation      **
** should failed due to the protection. This is displayed with LED7 turned on.**
*******************************************************************************/

#include "tle_device.h"
#include "eval_board.h"
#include "scu_defines.h"

/* Allocate one flash page in RAM */
__USED static uint8 TargetData[FlashPageSize];
__USED static int NVM_Error;
__USED static TCustomerID CustomerID;

int main(void)
{
  uint32 i;
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  /* Set the protection according to the design step */
  GetCustomerID((TCustomerID *) & CustomerID);

  switch (CustomerID.bit.Step)
  {
    case 0xAA:
      {
        /* UH design step */
        /* Enable write protection */
        USER_DFLASH_WR_PROT_EN(SCU_DFLASH_WPROT_PW);
      }
      break;
    case 0xAB:
      break;
    case 0xBA:
      {
        SCU_ChangeNVMProtection(NVM_DATA_WRITE, PROTECTION_SET);
      }
      break;
    case 0xBB:
      break;
    case 0xBC:
      break;
    case 0xBE:
      break;
    case 0xBF:
      {
        /* Enable write protection */
        USER_DFLASH_WR_PROT_EN(SCU_DFLASH_WPROT_PW);
      }
      break;
    default:
      break;
  }

  /* Fill up the TargetData array with housenumbers */
  for (i = 0; i < FlashPageSize; i++)
  {
    TargetData[i] = (uint8) i;
  }

  /* Call ProgramPage routine */
  /* If the write protection is enabled, the ProgramPage routine returns an error */
  NVM_Error = ProgramPage(DataFlashStart, TargetData, 0, 0, 0);

  /* Check if the programming worked fine */
  if (NVM_Error == 0u)
  {
    /* Data in page 0 of the Data Flash is correct */
    /* This is not expected in this example */
    PORT_ChangePin(LED1, PORT_ACTION_SET);
  }
  else
  {
    /* An error happened during ProgramPage */
    /* This is expected in this example as the data flash is protected */
    PORT_ChangePin(LED7, PORT_ACTION_SET);
  }

  for (;;)
  {
    (void)WDT1_Service();
  }
}
