/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** BG           Blandine Guillot                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2020-07-13, BG:   Initial version                                  **
** V1.1.0: 2021-07-19, BG:   EP-865: Updated example to be compatible with    **
**                           IAR compiler                                     **
** V1.2.1: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** NVM: Interrupts in RAM                                                     **
********************************************************************************
** In the chip, the Flash memory is single-port: during any operation         **
** (read/write/erase), no other operation is possible. While an NVM           **
** write/erase is ongoing, any interrupt (read operation) will be held until  **
** NVM is ready. The execution of the ISR will be postponed for the remaining **
** time of the NVM operation -> ISR will be delayed.                          **
** To have the interrupts still executed during a Flash operation, the        **
** solution consists in locating the interrupt vector table + the complete    **
** interrupt service routines in the RAM.                                     **
*******************************************************************************/

#include "tle_device.h"
#include "tle_variants.h"
#include <stdio.h>
#include <string.h>

/* address of the default vector table definition */
#define __Vectors_Size (0x80)

/* placing the interrupt vector table to the upper most RAM address */
#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION < 6000000)
  /* ARMCC V5 */
  extern uint32 __Vectors;
  #define IRQ_VECTOR_TABLE_RAM (RAMStart + RAMSize - __Vectors_Size)
  uint32 vtab[32] __attribute__((at(IRQ_VECTOR_TABLE_RAM), used));
  
#elif defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6000000)
  extern uint32 __Vectors;
  #define MAKE_STRING( string ) #string
  #define MAKE_SECTION_NAME( part1, part2 ) MAKE_STRING( part1##part2 )
  #define AT_ADDRESS_ZI( address ) MAKE_SECTION_NAME( .bss.ARM.__at_,address )

  #if (RAMSize == 0xC00u)
    #define IRQ_VECTOR_TABLE_RAM 0x18000B80u
  #elif (RAMSize == 0x1800u)
    #define IRQ_VECTOR_TABLE_RAM 0x18001780u
  #elif (RAMSize == 0x2000u)
    #define IRQ_VECTOR_TABLE_RAM 0x18001F80u
  #else
    #error RAM size not supported
  #endif /* RAMSize */
  uint32 vtab[32] __attribute__((section(AT_ADDRESS_ZI(IRQ_VECTOR_TABLE_RAM))));
#elif defined(__IAR_SYSTEMS_ICC__)
  extern uint32 __vector_table;
  #define IRQ_VECTOR_TABLE_RAM (RAMStart + RAMSize - __Vectors_Size)
//  #if (RAMSize == 0xC00u)
//    #pragma location=0x18000B80u
//  #elif (RAMSize == 0x1800u)
//    #pragma location=0x18001780u
//  #elif (RAMSize == 0x2000u)
//    #pragma location=0x18001F80u
//  #else
//    #error RAM size not supported
//  #endif /* RAMSize */
//  uint32 vtab[32];
#endif /* (__ARMCC_VERSION < 6000000) */

__USED static uint8 TargetData[FlashPageSize];
__USED static uint8 NVM_error;

int main(void)
{
  uint32 i;
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  TIMER2_Start();
  

  /* Fill up the TargetData array with housenumbers */
  for (i = 0; i < FlashPageSize; i++)
  {
    TargetData[i] = (uint8) i;
  }
  
  /* Copy first 128 bytes of flash containing the vector table to RAM */
  #if defined(__ARMCC_VERSION)
    memcpy((uint32_t *)IRQ_VECTOR_TABLE_RAM, (uint32_t *)&__Vectors, __Vectors_Size);
  #elif defined(__IAR_SYSTEMS_ICC__)
    memcpy((uint32_t *)IRQ_VECTOR_TABLE_RAM, (uint32_t *)&__vector_table, __Vectors_Size);
  #endif
  /* Set the Vector Table to the RAM address */
  CPU->VTOR.reg = (uint32)IRQ_VECTOR_TABLE_RAM;
  /* Toggle once P1.4 */
  PORT_P14_Output_Toggle_Set();
  /* Program NVM page */
  PORT_P04_Output_High_Set();
  /* NVM Program page with deactivated disable and enable interrupts */
  /* violation: cast from unsigned int to pointer [MISRA Rule 45]*/
  uint8 *pc = (uint8 *) DataFlashStart;
  /* trigger WDT1 Short-Open-Window */
  WDT1_SOW_Service(1u);
  /* open NVM assembly buffer */
  NVM_error = USER_OPENAB(DataFlashStart);
  /* trigger regular WDT1, close SOW */
  (void)WDT1_Service();

  /* check if OpenAB function returned a PASS (== 0) */
  if (NVM_error == (uint8)0)
  {
    /* fill up the assembly buffer with new data */
    for (i = 0u; i < (uint8)FlashPageSize; i++)
    {
      pc[i] = TargetData[i];
    }

    /* trigger WDT1 Short-Open-Window */
    WDT1_SOW_Service(1u);
    /* execute NVM programming function */
    NVM_error = USER_PROG((uint8)(((0 & (uint8)1) << 2u) | ((0 & (uint8)1) << 1u) | (0 & (uint8)1)));
    /* trigger regular WDT1, close SOW */
    (void)WDT1_Service();
  }

  if (NVM_error == 0u)
  {
    /* Data in the page 0 of Data Flash is correct */
    PORT_P04_Output_Low_Set();
  }
  else
  {
    /* An error happened during ProgramPage */
    PORT_P01_Output_High_Set();
  }

  for (;;)
  {
    /* Main watchdog1 (WDT1) service */
    (void)WDT1_Service();
  }
}
