/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** BG           Blandine Guillot                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2020-02-27, BG:   Initial version of revision history              **
** V1.0.2: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** PMU: Power saving - Evaluation of the reset source                         **
********************************************************************************
** Send the device into:                                                      **
**  - Sleep mode by pulling P1.1 to high                                      **
**  - Stop mode by pulling P1.0 to high                                       **
** The device wakes up by MON (configured by default).                        **
** Check the LEDs on the evaluation board to know which reset was performed:  **
**  - LED1 = Power On Reset                                                   **
**  - LED2 = PIN Reset                                                        **
**  - LED3 = Sleep Mode Exit                                                  **
**  - LED4 = Stop Mode Exit (exit with Reset only)                            **
**  - LED5 = System Fail                                                      **
*******************************************************************************/

#include "tle_device.h"
#include "eval_board.h"

void Evaluate_Reset_Source(void);

int main(void)
{
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  /* Evaluates the reset source and turns on a LED accordingly */
  Evaluate_Reset_Source();

  for (;;)
  {
    (void)WDT1_Service();

    if (PORT_ReadPin(0x11) == 1)
    {
      SCU_EnterSleepMode();
    }

    if (PORT_ReadPin(0x10) == 1)
    {
      SCU_EnterStopMode();
    }
  }
}

/* Evaluate the reset source and turns on a LED accordingly */
void Evaluate_Reset_Source(void)
{
  uint8 sts;
  sts = PMU_Get_Reset_Status();

  /* LED1 turned on in case of a Power On Reset */
  if (sts & PMU_RESET_STS_POR)
  {
    PORT_ChangePin(LED1, PORT_ACTION_SET);
  }

  /* LED2 turned on in case of a PIN Reset */
  if (sts & PMU_RESET_STS_PIN)
  {
    PORT_ChangePin(LED2, PORT_ACTION_SET);
  }

  /* LED3 turned on after a sleep mode exit */
  if (sts & PMU_RESET_STS_SLEEP)
  {
    PORT_ChangePin(LED3, PORT_ACTION_SET);
  }

  /* LED4 turned on after a stop mode exit */
  if (sts & PMU_RESET_STS_WAKE)
  {
    PORT_ChangePin(LED4, PORT_ACTION_SET);
  }

  /* LED5 turned on after a system fail */
  if (sts & PMU_RESET_STS_SYS_FAIL)
  {
    PORT_ChangePin(LED5, PORT_ACTION_SET);
  }

  PMU_Clear_Reset_Status();
}
