/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** BG           Blandine Guillot                                              **
** JO           Julia Ott                                                     **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.0.0: 2020-02-27, BG:   Initial version of revision history              **
** V1.0.1: 2021-07-19, BG:   Removed warnings for ARMCC v6 compiler           **
** V1.1.1: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** Timer2: Measurement of the PWM (duty cycle/frequency)                      **
********************************************************************************
** Two boards are necessary for this example:                                 **
**  - 1st board: flash this example -> master board                           **
**  - 2nd board: flash the CCU6_PWM example -> slave board         **
** Connect P1.2 of the master board with one of the PWM output pins of the    **
** slave board.                                                               **
**                                                                            **
** Measuring the low part of the PWM:                                         **
** Timer2 starts counting with a falling edge on P1.2                         **
** Timer2 captures on the rising edge on P1.2 (PWM low phase, capDC)          **
**                                                                            **
** Measuring the period of the PWM:                                           **
** Timer2 still runs from previous falling edge                               **
** Timer2 captures on the falling edge on P1.2 (PWM period, capPer)           **
**                                                                            **
** Always one period is skipped in order to reprogram the edge detection of   **
** the Timer2.                                                                **
**                                                                            **
** Max. PWM period is 1.6ms (625Hz)                                           **
** @ 1kHz PWM frequency (1ms) the min. recognizable DC is 0.3%, max. 99.8%    **
********************************************************************************
**         start          capture   capture                capture            **
**         Timer2         Timer2    Timer2                 Timer2             **
**       -----                --------              ---------                 **
** P1.2       |              |        |            |         |                **
**            :--------------:\       :------------          :------...       **
**            :    capDC     : \      :\                     :    capDC       **
**            :<------------>:  \     : \                    :<-----...       **
**            :                  \    :  \                   :                **
**            :     PWM_Capture() ISR :   \                  :                **
**           (captured on rising edge):    \                 :                **
**            :                       :     \                :                **
**            :    capPer             :      \               :    capPer      **
**            :<--------------------->:       \              :<-----...       **
**                                             \                              **
**                                      PWM_Capture() ISR                     **
**                            (captured on  falling edge)                     **
*******************************************************************************/

#include "tle_device.h"
#include "eval_board.h"

__USED static uint16 capPer, capDC;
__USED static uint16 bDC_Per;

void PWM_Capture(void);


int main(void)
{
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  bDC_Per = 0;

  while (1)
  {
    (void)WDT1_Service();
  }
}


/* Callback function for Timer2 */
void PWM_Capture(void)
{
  /* Timer2 in run mode */
  if (TIMER2->T2CON.bit.TR2 == 1u)
  {
    if (bDC_Per == 0u)
    {
      /* Get the duty cycle */
      capDC = TIMER2_Get_Capture();
      /* Next capture on falling edge */
      TIMER2->T2MOD.bit.EDGESEL = 0u;
      bDC_Per = 1u;
    }
    else
    {
      /* Get the PWM period */
      capPer = TIMER2_Get_Capture();
      /* Next capture on rising edge */
      TIMER2->T2MOD.bit.EDGESEL = 1u;
      bDC_Per = 0u;
      /* Stop Timer2 */
      TIMER2_Stop();
      /* Reset Timer2 */
      TIMER2_Clear_Count();
    }
  }
}
