/*
 ***********************************************************************************************************************
 *
 * Copyright (c) Infineon Technologies AG
 * All rights reserved.
 *
 * The applicable license agreement can be found at this pack's installation directory in the file
 * license/IFX_SW_Licence_MOTIX_LITIX.txt
 *
 **********************************************************************************************************************/

/*******************************************************************************
**                             Author(s) Identity                             **
********************************************************************************
** Initials     Name                                                          **
** ---------------------------------------------------------------------------**
** JO           Julia Ott                                                     **
** BG           Blandine Guillot                                              **
*******************************************************************************/

/*******************************************************************************
**                          Revision Control History                          **
********************************************************************************
** V1.3.0: 2020-02-28, JO:   Initial version of the example, published in     **
**                           TLE986x Pack 1.4.4                               **
** V1.3.1: 2021-07-19, BG:   Removed warnings for ARMCC v6 compiler           **
** V1.4.1: 2025-03-13, JO:   EP-1580: Aligned example with TLE987x code       **
*******************************************************************************/

/*******************************************************************************
**                                  Abstract                                  **
********************************************************************************
** Timer3: Mode1B - 16-bit timer triggered by an event                        **
********************************************************************************
** CCU6 Ch0 generates a pwm signal at P0.4 with a frequency of 10kHz and a    **
** duty cycle of 50%.                                                         **
**                                                                            **
** At every rising edge of the pwm signal, Timer3 is reset and                **
** starts counting.                                                           **
** After 10us, Timer3 triggers an ADC1 measurement and an interrupt.          **
** To visualize the interrupt, P1.1 is set to high in the interrupt.          **
** When the ADC1 measurement has finished, an ADC1 EOC interrupt is           **
** generated ('Poti_Handler') which sets P1.1 to low and visualizes the       **
** measured value on the LEDs.                                                **
**                                                                            **
********************************************************************************
**           ____________________                    ___________________      **
** P0.4     |                    |                  |                   ...   **
** CC60 ----                      ------------------                          **
**          : reset & start                         : reset & start           **
**          : Timer3 (10us)                         : Timer3 (10us)           **
**          :         _                             :         _               **
** ADC1 Trig:        | |                            :        | |              **
**      ----:--------:  ----------------------------:--------:  ---------     **
**          :<------>:                              :<------>:                **
**             10us  :                                 10us  :                **
** ADC1 Meas         :<------>:                              :<------>:       **
**                   : 1.4us  :                              : 1.4us  :       **
**                   :        :_                             :        :_      **
** ADC1 EOC          :        | |                            :        | |     **
**      -------------:--------:  ----------------------------:--------:  -    **
**                   :________:                              :________:       **
** P1.1              |        |                              |        |       **
**      -------------          ------------------------------          -...   **
** Toggling of P1.1 is delayed due to interrupt latency (enter interrupt,     **
** return from interrupt), ISR handling, and Port toggling                    **
*******************************************************************************/

#include "tle_device.h"

void Timer3_HB_Ovf_INT(void);
void Poti_Handler(void);


int main(void)
{
  /* Initialization of hardware modules based on Config Wizard configuration */
  TLE_Init();
  /* Start Timer3 */
  TIMER3_Start();
  /* Start PWM generation */
  CCU6_T12_Start();

  for (;;)
  {
    (void)WDT1_Service();
  }
}


/* Callback function for each overflow of Timer3 */
void Timer3_HB_Ovf_INT(void)
{
  PORT_P11_Output_High_Set();
}


void Poti_Handler(void)
{
#define Interval_mV 625u
  uint16 mV;
  bool value_valid;
  PORT_P11_Output_Low_Set();
  value_valid = ADC1_GetChResult_mV(&mV, ADC1_CH4);

  if (value_valid == true)
  {
    switch (mV / Interval_mV)
    {
      case 0:
        PORT_P01_Output_Low_Set();
        PORT_P02_Output_Low_Set();
        PORT_P03_Output_Low_Set();
        PORT_P12_Output_Low_Set();
        PORT_P10_Output_Low_Set();
        PORT_P13_Output_Low_Set();
        PORT_P14_Output_Low_Set();
        break;
      case 1:
        PORT_P01_Output_High_Set();
        PORT_P02_Output_Low_Set();
        PORT_P03_Output_Low_Set();
        PORT_P12_Output_Low_Set();
        PORT_P10_Output_Low_Set();
        PORT_P13_Output_Low_Set();
        PORT_P14_Output_Low_Set();
        break;
      case 2:
        PORT_P01_Output_High_Set();
        PORT_P02_Output_High_Set();
        PORT_P03_Output_Low_Set();
        PORT_P12_Output_Low_Set();
        PORT_P10_Output_Low_Set();
        PORT_P13_Output_Low_Set();
        PORT_P14_Output_Low_Set();
        break;
      case 3:
        PORT_P01_Output_High_Set();
        PORT_P02_Output_High_Set();
        PORT_P03_Output_High_Set();
        PORT_P12_Output_Low_Set();
        PORT_P10_Output_Low_Set();
        PORT_P13_Output_Low_Set();
        PORT_P14_Output_Low_Set();
        break;
      case 4:
        PORT_P01_Output_High_Set();
        PORT_P02_Output_High_Set();
        PORT_P03_Output_High_Set();
        PORT_P12_Output_High_Set();
        PORT_P10_Output_Low_Set();
        PORT_P13_Output_Low_Set();
        PORT_P14_Output_Low_Set();
        break;
      case 5:
        PORT_P01_Output_High_Set();
        PORT_P02_Output_High_Set();
        PORT_P03_Output_High_Set();
        PORT_P12_Output_High_Set();
        PORT_P10_Output_High_Set();
        PORT_P13_Output_Low_Set();
        PORT_P14_Output_Low_Set();
        break;
      case 6:
        PORT_P01_Output_High_Set();
        PORT_P02_Output_High_Set();
        PORT_P03_Output_High_Set();
        PORT_P12_Output_High_Set();
        PORT_P10_Output_High_Set();
        PORT_P13_Output_High_Set();
        PORT_P14_Output_Low_Set();
        break;
      default:
        PORT_P01_Output_High_Set();
        PORT_P02_Output_High_Set();
        PORT_P03_Output_High_Set();
        PORT_P12_Output_High_Set();
        PORT_P10_Output_High_Set();
        PORT_P13_Output_High_Set();
        PORT_P14_Output_High_Set();
        break;
    }		
  }
}
